<?php

namespace App\Models;

use App\Models\State;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Builder;

class Address extends BaseModel
{
    use HasFactory;

    /**
     * The database table used by the model.
     *
     * @var string
     */
    protected $table = 'addresses';

    /**
     * The relations to eager load on every query.
     *
     * @var array
     */
    protected $with = ['country:id,name,iso_code', 'state:id,name,country_id,iso_code'];
    // protected $with = ['country:id,name,country_code,iso_code', 'state:id,name,country_id,iso_code'];

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'address_title',
        'address_type',
        'address_line_1',
        'address_line_2',
        'city',
        'state_id',
        'country_id',
        'zip_code',
        'phone',
        'addressable_id',
        'addressable_type',
        'latitude',
        'longitude',
    ];

    /**
     * Get all of the owning addressable models.
     */
    public function addressable()
    {
        return $this->morphTo();
    }

    /**
     * Get the country of the address.
     */
    public function country()
    {
        return $this->belongsTo(Country::class);
    }

    /**
     * Get the state of the address.
     */
    public function state()
    {
        return $this->belongsTo(State::class);
    }

    /**
     * Creat new country and set the id if the given value is not available
     */
    public function setCountryIdAttribute($value)
    {
        if (!is_numeric($value) && $value != null) {
            $country = Country::select('id')->where('iso_code', $value)->first();

            if (!$country) {
                $country = Country::select('id')->where('name', $value)->first();
            }

            if (!$country) {
                $country = Country::select('id')->where('iso_numeric', $value)->first();
            }

            if (!$country) {
                $country = Country::select('id')->where('full_name', $value)->first();
            }

            if (!$country) {
                throw new \Exception(trans('theme.country_not_exist'), 404);
            }

            $value = $country?->id;
        }

        $this->attributes['country_id'] = $value;
    }

    /**
     * Creat new state and set the id if the given value is not available
     */
    public function setStateIdAttribute($value)
    {
        if (!is_numeric($value) && $value != null) {
            // Getting the country id
            if (isset($this->attributes['country_id']) && is_numeric($this->attributes['country_id'])) {
                $country_id = $this->attributes['country_id'];
            } else {
                $country_id = \Request::input('country_id');
            }

            $state = State::select('id')
                ->where('country_id', $country_id)->where(
                    function (Builder $q) use ($country_id, $value) {
                        $q->where('name', $value)
                            ->orWhere('iso_code', $value);
                    }
                )->first();

            // Create state
            if (!$state) {
                $state = State::create([
                    'country_id' => $country_id,
                    'name' => $value
                ]);
            }

            $value = $state->id;
        }

        $this->attributes['state_id'] = $value;
    }

    /**
     * Try to fetch the coordinates from Google
     * and store it to database
     *
     * @return $this
     */
    public function toGeocodeString()
    {
        $data = [];
        $data[] = $this->address_line_1 ?? '';
        $data[] = $this->address_line_2 ?? '';
        $data[] = $this->city ?? '';
        if ($this->state_id) {
            $data[] = $this->state?->name;
        }
        $data[] = $this->zip_code ?? '';
        if ($this->country_id) {
            $data[] = $this->country?->name;
        }

        // build str string
        $str = trim(implode(', ', array_filter($data)));

        return str_replace(' ', '+', $str);
    }

    /**
     * Formate the address toHtml
     *
     * @param  string $separator html code
     *
     * @return string
     */
    public function toHtml($separator = '<br/>', $show_type = true)
    {
        $html = [];

        if (Customer::class == $this->addressable_type && $show_type) {
            $html[] = '<strong class="pull-right">' . strtoupper($this->address_type) . '</strong>';
        }

        if (config('system_settings.show_address_title')) {
            $html[] = $this->address_title;
        }

        if (strlen($this->address_line_1)) {
            $html[] = $this->address_line_1;
        }

        if (strlen($this->address_line_2)) {
            $html[] = $this->address_line_2;
        }

        if (strlen($this->city)) {
            $html[] = $this->city . ', ';
        }

        if (strlen($this->state_id) || $this->zip_code) {
            $html[] = sprintf('%s %s', e($this->state_id ? optional($this->state)->name : ''), e($this->zip_code));
        }

        if (config('system_settings.address_show_country') && $this->country) {
            $html[] = e($this->country?->name);
        }

        if (strlen($this->phone)) {
            $html[] = '<abbr title="' . trans('app.phone') . '">P:</abbr> ' . e($this->phone);
        }

        $addressStr = implode($separator, $html);

        $return = '<address>' . $addressStr . '</address>';

        return $return;
    }

    /**
     * Return a "string formatted" version of the address
     */
    public function toString($title = false)
    {
        $str = [];

        if ($title || config('system_settings.show_address_title')) {
            $str[] = $this->address_title;
        }

        if (strlen($this->address_line_1)) {
            $str[] = $this->address_line_1;
        }

        if (strlen($this->address_line_2)) {
            $str[] = $this->address_line_2;
        }

        if (strlen($this->city)) {
            $str[] = $this->city;
        }

        if (strlen($this->state_id) || $this->zip_code) {
            $str[] = sprintf('%s %s', e($this->state_id ? $this->state?->name : ''), e($this->zip_code));
        }

        // if (strlen($this->city)) {
        //     $state_name = $this->state ? $this->state?->name : '';
        //     $str []= sprintf('%s, %s %s', $this->city, $state_name, $this->zip_code);
        // }

        if (config('system_settings.address_show_country') && $this->country) {
            $str[] = $this->country?->name;
        }

        if (strlen($this->phone)) {
            $str[] = trans('app.phone') . ': ' . e($this->phone);
        }

        return implode(', ', $str);
    }

    /**
     * Return a city and state string of the address
     */
    public function toShortString()
    {
        $str = [];

        if (strlen($this->city)) {
            $str[] = $this->city;
        }

        if (strlen($this->state_id)) {
            $str[] = $this->state?->name;
        }

        return implode(', ', $str);
    }

    /**
     * Get address as array
     *
     * @return array|null
     */
    public function toArray()
    {
        $address = [];
        $address['address_type'] = $this->address_type;
        $address['address_title'] = $this->address_title;
        $address['address_line_1'] = $this->address_line_1;
        $address['address_line_2'] = $this->address_line_2;
        $address['city'] = $this->city;

        if ($this->state) {
            $address['state'] = $this->state?->name;
        }

        $address['zip_code'] = $this->zip_code;

        if ($this->country) {
            $address['country'] = $this->country?->name;
        }

        $address['phone'] = $this->phone;

        if ($this->latitude && $this->longitude) {
            $address['latitude'] = $this->latitude;
            $address['longitude'] = $this->longitude;
        }

        return !empty($address) ? array_filter($address) : null;
    }

    /**
     * Get address for stripe.
     *
     * @return array|null
     */
    public function toStripeAddress()
    {
        $address = [];
        $address['line1'] = $this->address_line_1;
        $address['line2'] = $this->address_line_2;
        $address['postal_code'] = $this->zip_code;
        $address['city'] = $this->city;

        if ($this->state) {
            $address['state'] = $this->state?->iso_code;
        }

        if ($this->country) {
            $address['country'] = $this->country?->iso_code;
        }

        return !empty($address) ? array_filter($address) : null;
    }
}
